/*==============================================================================
SUMMARY STATISTICS BANK VALUATIONS TABLE
==============================================================================
Purpose:
This script generates summary statistics tables for different components of
bank value for both all banks and large banks. 

Input:
- $path_clean/analysis_ready_data.dta (Master dataset)
- Global variables: $tab_valuations_a, $tab_valuations_b (main analysis output filenames)
- Global variables: $tab_valuations_ext_a, $tab_valuations_ext_b (extension output filenames)
- Global variable: ${ext_suffix} (determines if running for extension)

Output:
- $path_tables/${tab_valuations_a}.tex (Main analysis Panel A)
- $path_tables/${tab_valuations_b}.tex (Main analysis Panel B)
- $path_tables/${tab_valuations_ext_a}.tex (Extension Panel A)
- $path_tables/${tab_valuations_ext_b}.tex (Extension Panel B)

Paper Correspondence:
- Main Analysis: Table 2 Panel A and Panel B
- Extensions: Tables A.6 - A.9 (Panels A and B)

Last updated: July 03, 2025
==============================================================================*/

display "--- Creating Summary Statistics Bank Valuations Table ---"

//===============================================================================
// Step 1: Prepare Variables for Summary Statistics
//===============================================================================
// Purpose: Calculate the total number of banks and the number/percentage of
// banks with negative bank value for each period and bank group.
local periods `" "dec2021" "feb2023" "feb2024" "'

// Calculate total number of banks and number/percentage with negative values for All Banks
foreach period_name in `periods' {
    * Count total observations for the current period and sample selection criteria
    gen n = 1 if period == "`period_name'" 
    bys period: egen tot_`period_name' = total(n)

    * Calculate number and percentage of banks with negative values for each bank value measure
    foreach v_base in v_noDF v_0 v_1 {
        gen temp = 0
        replace temp = 1 if `v_base' < 0 & period == "`period_name'" 
        bys period: egen temp2 = total(temp)
        * Percentage of banks with negative value
        gen `v_base'_`period_name'_neg = temp2 * 100 / tot_`period_name' if period == "`period_name'" 
        * Number of banks with negative value
        gen `v_base'_`period_name'_neg_num = temp2 if period == "`period_name'"
        drop temp*
    }
    drop n
}

// Calculate total number of banks and number/percentage with negative values for Large Banks
foreach period_name in `periods' {
    * Count total observations for the current period, sample selection, and large bank criteria
    gen n = 1 if period == "`period_name'" & assets_bill >= 100
    bys period: egen tot_`period_name'_big = total(n)

    * Calculate number and percentage of banks with negative values for each bank value measure (Large Banks)
    foreach v_base in v_noDF v_0 v_1 {
        gen temp = 0
        replace temp = 1 if `v_base' < 0 & assets_bill >= 100 & period == "`period_name'"
        bys period: egen temp2 = total(temp)
        * Percentage of large banks with negative value
        gen `v_base'_`period_name'_neg_big = temp2 * 100 / tot_`period_name'_big if period == "`period_name'" & assets_bill >= 100
        * Number of large banks with negative value
        gen `v_base'_`period_name'_neg_big_num = temp2 if period == "`period_name'" & assets_bill >= 100
        drop temp*
    }
    drop n
}

//-------------------------------------------------------------------------------
// Panel D: Bank Value Summary Statistics, All Banks (using texdoc)
//-------------------------------------------------------------------------------
// Determine output filename based on context (main analysis or extension)
local output_filename_a = ""
if "${ext_suffix}" != "" {
    local output_filename_a = "${tab_valuations_ext_a}"
}
else {
    local output_filename_a = "${tab_valuations_a}"
}

// Initialize texdoc for the All Banks table
texdoc init "$path_tables/`output_filename_a'.tex", replace force

// Write LaTeX table header
tex {\def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}
tex \begin{tabular*}{.9\textwidth}{@{\hskip\tabcolsep\extracolsep\fill}l*{3}{c}}
tex \hline \hline
tex &\multicolumn{3}{c}{All banks}
tex \\\cmidrule(lr){2-4}
tex Bank Value&\multicolumn{1}{c}{Dec 2021}&\multicolumn{1}{c}{Feb 2023}&\multicolumn{1}{c}{Feb 2024}
tex \\\cmidrule(lr){2-2}\cmidrule(lr){3-3}\cmidrule(lr){4-4}
tex &\multicolumn{1}{c}{(1)}   &\multicolumn{1}{c}{(2)} &\multicolumn{1}{c}{(3)} \\
tex \hline

// Collect summary statistics into local macros
foreach period_name in `periods' { 


    * Get total observations for the period
    summarize rssdid if period=="`period_name'" & !mi(v_0)
    local n_`period_name': di %6.0f r(N)

    foreach v in v_noDF v_0 v_1 { // Loop through bank value variables
        /* display "`period_name'; `sample_cond'" */
        * Summarize the main bank value variable
        summarize `v' if period=="`period_name'" & !mi(v_0)
        local mean_`v'_`period_name': di %6.2f r(mean)
        local sd_`v'_`period_name': di %6.2f r(sd)
        * Remove leading/trailing spaces from standard deviation for LaTeX
        local sd_`v'_`period_name' = subinstr("`sd_`v'_`period_name''", " ", "", .)

        * Summarize the percentage of banks with negative value
        summarize `v'_`period_name'_neg if period=="`period_name'" & !mi(v_0)
        local mean_`v'_`period_name'_neg: di %6.2f r(mean)
    }
}

// Write table rows using collected local macros
foreach v in v_noDF v_0 v_1 { // Loop through bank value variables for rows

    * Define LaTeX label for the current bank value measure
    if "`v'"=="v_noDF"{
        loc vlab \$ A - D $ (No DF)
    }
    else if "`v'"=="v_0"{
        loc vlab \$ V(0,r) = A - D + DF_I $ (Run)
    }
    else if "`v'"=="v_1"{
        loc vlab \$ V(1,r) = A - D + DF_I+ DF_U $ (No run)
    }

    * Print row for Mean value
    tex `vlab' & `mean_`v'_dec2021' & `mean_`v'_feb2023' & `mean_`v'_feb2024' \\
    * Print row for Standard Deviation
    tex & (`sd_`v'_dec2021') & (`sd_`v'_feb2023') & (`sd_`v'_feb2024') \\
    * Print row for Percentage of banks with negative value
    tex \$ \% \le 0 $ & `mean_`v'_dec2021_neg' & `mean_`v'_feb2023_neg' & `mean_`v'_feb2024_neg' \\

    * Add vertical space between blocks of variables, except after the last one
    if "`v'"!="v_1"{
        tex [1em]
    }
}

// Write table footer (Observations and closing)
tex \hline
tex Obs. & `n_dec2021' & `n_feb2023' & `n_feb2024' \\
tex \hline\hline \end{tabular*}}

// Close texdoc file
texdoc close

display "--- Table 2, Panel a created: `output_filename_a'.tex ---"

//-------------------------------------------------------------------------------
// Panel E: Bank Value Summary Statistics, Large Banks (using texdoc)
//-------------------------------------------------------------------------------
// Determine output filename based on context (main analysis or extension)
local output_filename_b = ""
if "${ext_suffix}" != "" {
    local output_filename_b = "${tab_valuations_ext_b}"
}
else {
    local output_filename_b = "${tab_valuations_b}"
}

// Initialize texdoc for the Large Banks table
texdoc init "$path_tables/`output_filename_b'.tex", replace force

// Write LaTeX table header
tex {\def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}
tex \begin{tabular*}{.9\textwidth}{@{\hskip\tabcolsep\extracolsep\fill}l*{3}{c}}
tex \hline \hline
tex &\multicolumn{3}{c}{Large banks}
tex \\\cmidrule(lr){2-4}
tex Bank Value&\multicolumn{1}{c}{Dec 2021}&\multicolumn{1}{c}{Feb 2023}&\multicolumn{1}{c}{Feb 2024}
tex \\\cmidrule(lr){2-2}\cmidrule(lr){3-3}\cmidrule(lr){4-4}
tex &\multicolumn{1}{c}{(1)}   &\multicolumn{1}{c}{(2)} &\multicolumn{1}{c}{(3)} \\
tex \hline

// Collect summary statistics into local macros
foreach period_name in `periods' { 

    * Define sample condition including large bank criteria
    local sample_cond "period=="`period_name'" & !mi(v_0) & assets_bill>=100"

    * Get total observations for the period (Large Banks)
    summarize rssdid if `sample_cond'
    local n_`period_name': di %6.0f r(N)

    foreach v in v_noDF v_0 v_1 { // Loop through bank value variables
        * Summarize the main bank value variable (Large Banks)
        summarize `v' if `sample_cond'
        local mean_`v'_`period_name': di %6.2f r(mean)
        local sd_`v'_`period_name': di %6.2f r(sd)
        * Remove leading/trailing spaces from standard deviation for LaTeX
        local sd_`v'_`period_name' = subinstr("`sd_`v'_`period_name''", " ", "", .)

        * Summarize the NUMBER of large banks with negative value
        summarize `v'_`period_name'_neg_big_num if `sample_cond'
        local mean_`v'_`period_name'_neg: di %6.0f r(mean) // Display as integer count
    }
}

// Write table rows using collected local macros
foreach v in v_noDF v_0 v_1 { // Loop through bank value variables for rows

    * Define LaTeX label for the current bank value measure
    if "`v'"=="v_noDF"{
        loc vlab \$ A - D $ (No DF)
    }
    else if "`v'"=="v_0"{
        loc vlab \$ V(0,r) = A - D + DF_I $ (Run)
    }
    else if "`v'"=="v_1"{
        loc vlab \$ V(1,r) = A - D + DF_I+ DF_U $ (No run)
    }

    * Print row for Mean value
    tex `vlab' & `mean_`v'_dec2021' & `mean_`v'_feb2023' & `mean_`v'_feb2024' \\
    * Print row for Standard Deviation
    tex & (`sd_`v'_dec2021') & (`sd_`v'_feb2023') & (`sd_`v'_feb2024') \\
    * Print row for NUMBER of banks with negative value
    tex \$ \# \le 0 $ & `mean_`v'_dec2021_neg' & `mean_`v'_feb2023_neg' & `mean_`v'_feb2024_neg' \\

    * Add vertical space between blocks of variables, except after the last one
    if "`v'"!="v_1"{
        tex [1em]
    }
}

// Write table footer (Observations and closing)
tex \hline
tex Obs. & `n_dec2021' & `n_feb2023' & `n_feb2024' \\
tex \hline\hline \end{tabular*}}

// Close texdoc file
texdoc close

display "--- Table 2, panel b created: `output_filename_b'.tex ---"

display "--- Finished Summary Statistics Bank Valuations Table ---"
